# =======================================================================
# phaser_MR.tcl --
#
# Maximum Likelihood Molecular Replacement
#
# CCP4Interface
#
# =======================================================================

#------------------------------------------------------------------------------
proc phaser_MR_prereq { } {
#------------------------------------------------------------------------------
  if { ![file exists [FindExecutable phaser]] } {
    return 0
  }
  return 1
}

#------------------------------------------------------------------------
  proc phaser_MR_setup { typedefVar arrayname } {
#------------------------------------------------------------------------
  upvar #0 $typedefVar typedef
  upvar #0 $arrayname array

  DefineMenu _mode_menu [list "automated search" \
                              "rotation search" \
                              "translation search" \
                              "refinement and phasing" \
                              "log-likelihood gain rescoring" \
                              "packing" \
                              "anisotropy correction" \
                              "cell content analysis" ] \
                        [list "AUTO" \
                              "ROT" \
                              "TRA" \
                              "RNP" \
                              "LLG" \
                              "PAK" \
                              "ANO" \
                              "CCA" ]

  DefineMenu _mode_rot \
    [list "fast rotation function" "brute rotation function" ] [list "MR_FRF" "MR_BRF"]

  DefineMenu _mode_tran \
    [list "fast translation function" "brute translation function" ] [list "MR_FTF" "MR_BTF" ]

  DefineMenu _spacegroup_option \
    [list "mtz space group" "mtz space group and enantiomorph" "all alternative space groups" "alternative spacegroups (listed)"] \
    [list "SG_MTZ" "SG_ENANT" "SG_ALL" "SG_ALT" ]

  set typedef(_laue_space_group) { varmenu LAUE_SPGP_LIST LAUE_SPGP_ALIAS 8 }

  DefineMenu _composition_option \
    [ list "average solvent content for protein crystal" "solvent content of protein crystal" "components in asymmetric unit" "fraction scattering by ensemble"] \
    [ list "AVERAGE" "SOLVENT" "COMPONENT" "FRAC"]

  DefineMenu _protnucleic \
    [list "protein" "nucleic acid" ] [list "PROTEIN" "NUCLEIC" ]

  DefineMenu _rmsid_option \
   [list "rms difference" "sequence identity" ] [list "RMS" "IDENT" ]

  set typedef(_mwseqnres_option) \
    { menu  { "molecular weight"  "sequence file" "number of residues"} { "MW"  "FASTA" "NRES"} }

  DefineMenu _ensemble_option \
    [list "pdb file(s)" "map (mtz file)" ] [list "PDB"  "MAP"]
 
  DefineMenu _search_type \
    [list "on" "off" ] [list "SEARCH_OR" "SEARCH_SIMPLE"]

  DefineMenu _search_method \
    [list "full" "fast" ] [list "FULL" "FAST" ]

  DefineMenu _brf_search_option \
   [list "all angles" "around an angle" ] [list "FULL" "AROUND" ]

  DefineMenu _btf_search_option \
   [list "full" "around" "region" "line"] [list "FULL" "AROUND" "REGION" "LINE"]

  DefineMenu _packing_option \
    [list "best packing (up to percent)" "best packing (up to number)" "max number of clashes" "accept all solutions" ] \
    [list "PERCENT" "BEST" "ALLOWED" "ALL"]

  set typedef(_ensemble_menu)  \
    { varmenu ENSEMBLE_MENU_LIST ENSEMBLE_MENU_ALIAS 15 }
  set typedef(_frf_target_option) \
    { menu { "LERF1" "LERF2" "Crowther" } { "LERF1" "LERF2" "CROWTHER" } }
  set typedef(_ftf_target_option) \
    { menu { "LETF1" "Correlation" } { "LETF1" "CORREL" } }
  set typedef(_peaks_option) \
   { menu  { "percentage of top peak" "sigma (Z-score) cutoff" "number of peaks" "all peaks"} \
           { "PERCENT" "SIGMA" "NPEAKS" "ALL" } }
  set typedef(_macro_protocol) \
    { menu  { "default"  "none (fix)"  "custom" "all (ref)"} { "DEFAULT" "OFF" "CUSTOM" "ALL"} }
  set typedef(_onoff) \
    { menu { "on" "off" } { ON OFF } }
  set typedef(_coord_option) \
    { menu { "orthogonal" "fractional" } { "ORTH" "FRAC" } }

  return 1
  }

#------------------------------------------------------------------------
proc phaser_MR_run { arrayname } {
#------------------------------------------------------------------------

  upvar #0 $arrayname array

# The output file array is set up in phaser_MR.script
  set array(OUTPUT_FILES) ""

# All output files with the user specified root will be saved to the project dir
#set array(FILENAME_ROOT) [FileJoin [GetDefaultDirPath] $array(ROOT) ]

  if { $array(HKLIN_MAIN) == "" } {
     WarningMessage "NOT SET: mtz reflection data file"
     return 0
  }

  if { ([GetValue $arrayname SCATTERING] == "COMPONENT") } {
  for { set n 1 } { $n <= $array(N_COMPONENT) } { incr n } {
    if { $array(COMP_OPTION,$n) == "sequence file" &&  $array(COMP_FILE,$n) == "" } {
       WarningMessage "NOT SET: sequence file for component # $n"
       return 0
     }
    if { $array(COMP_OPTION,$n) == "molecular weight" &&  $array(MW,$n) == "" } {
       WarningMessage "NOT SET: molecular weight for component # $n"
       return 0
     }
   } }

  if { ([GetValue $arrayname SCATTERING] == "NRES")} { 
    if { ([GetValue $arrayname COMP_NRES] == "")} { 
     WarningMessage "NOT SET: number of residues in the composition"
     return 0
    } }

  if { ([GetValue $arrayname SCATTERING] == "SOLVENT")} { 
    if { ([GetValue $arrayname SCATTERING_SOLVENT] == "")} { 
     WarningMessage "NOT SET: solvent content in the composition"
     return 0
  } }

  if { ([GetValue $arrayname MR_MODE] == "ANO") } {
     return 1
  }

  if { ([GetValue $arrayname MR_MODE] == "CCA") } {
     return 1
  }

   for { set n 1 } { $n <= $array(N_ENSEMBLE) } { incr n } {
    if { [ regexp PDB [GetValue $arrayname ENSEMBLE_OPTION,$n ] ] } {
    for { set p 1 } { $p <= $array(N_PDB,$n) } { incr p } {
      if { $array(XYZIN,[subst $n]_$p) == "" } {
         WarningMessage "NOT SET: pdb file for ensemble # $n"
         return 0
      }
      if { $array(RMS,[subst $n]_$p) == "" } {
         WarningMessage "NOT SET: RMS error for PDB # $p of ensemble # $n"
         return 0
      }
    }
    }
    if { [ regexp MAP [GetValue $arrayname ENSEMBLE_OPTION,$n ] ] } {
      if { $array(HKLIN,$n) == "" } {
         WarningMessage "NOT SET: mtz map file for ensemble # $n"
         return 0
      }
      if { $array(ENSEMBLE_PROTMW,$n) == "" } {
         WarningMessage "NOT SET: protein molecular weight for ensemble # $n"
         return 0
      }
      if { $array(ENSEMBLE_NUCLMW,$n) == "" } {
         WarningMessage "NOT SET: nucleic acid molecular weight for ensemble # $n"
         return 0
      }
      if { ($array(ENSEMBLE_EXT_X,$n) == "") 
        || ($array(ENSEMBLE_EXT_Y,$n) == "")
        || ($array(ENSEMBLE_EXT_Z,$n) == "") } {
         WarningMessage "NOT SET: extent of the model for ensemble # $n"
         return 0
      } 
      if { $array(ENSEMBLE_RMS_MTZ,$n) == "" } {
         WarningMessage "NOT SET: RMS error of the model for ensemble # $n"
         return 0
      } 
      if { ($array(ENSEMBLE_CEN_X,$n) == "")
        || ($array(ENSEMBLE_CEN_Y,$n) == "")
        || ($array(ENSEMBLE_CEN_Z,$n) == "") } {
         WarningMessage "NOT SET: centre of the model for ensemble # $n"
         return 0 
       }
    }
   }

# Check for compulsory brute force rotation function AROUND search parameters
  if { ([GetValue $arrayname MR_MODE] == "ROT") 
    && ([GetValue $arrayname MR_MODE_SELECT] == "MR_BRF")
    && ([GetValue $arrayname BRF_SEARCH_OPTION] == "AROUND") } { 
    if { ($array(AROUND_ALPHA) == "") || ($array(AROUND_BETA) == "")
      || ($array(AROUND_GAMMA) == "") || 
         ($array(AROUND_EULER_RANGE) == "") } {
         WarningMessage "NOT SET: rotation function search (euler angles and range)"
         return 0
    }
   }

if { (([GetValue $arrayname MR_ROT_SEARCH] == "SEARCH_SIMPLE") 
     && ( [GetValue $arrayname MR_MODE] == "ROT") 
     && ( !($array(N_SIMPLE_SEARCH) == 0 && ($array(N_ENSEMBLE) == 0))) ) } {
if { ($array(MR_ROT_SEARCH_ID) == "") } {
    WarningMessage "NOT SET: search ensemble" 
    return 0
    }
if { ([lsearch -exact $array(ENSEMBLE_MENU_LIST) $array(MR_ROT_SEARCH_ID)] < 0) } {
    WarningMessage "The search ensemble is not in the defined list of ensembles"
    return 0
    }
} 

if { (([GetValue $arrayname MR_SEARCH] == "SEARCH_SIMPLE") 
     && ( [GetValue $arrayname MR_MODE] == "AUTO") ) } {
if { $array(N_SIMPLE_SEARCH) < 1 } {
if { $array(N_ENSEMBLE) > 0 } {
    WarningMessage "NOT SET: search ensemble"
    return 0
} } }

if { (([GetValue $arrayname MR_SEARCH] == "SEARCH_SIMPLE") 
     && ( [GetValue $arrayname MR_MODE] == "AUTO") ) } {
if { $array(N_SIMPLE_SEARCH) < 1 } {
if { $array(N_ENSEMBLE) < 1 } {
if { [GetValue $arrayname SET_SOL_FILE_NOTOB] == "" } {
    WarningMessage "NOT SET: ensembles and searches must be defined in SOL file"
    return 0
} } } }


if { (([GetValue $arrayname MR_SEARCH] == "SEARCH_SIMPLE") 
     && ( [GetValue $arrayname MR_MODE] == "AUTO")
     && ( !($array(N_SIMPLE_SEARCH) == 0 && ($array(N_ENSEMBLE) == 0))) ) } {
for { set q 1 } { $q <= $array(N_SIMPLE_SEARCH) } { incr q } {
if { ($array(MR_SIMPLE_ENSEMBLE_ID,$q) == "") } {
    WarningMessage "NOT SET: search ensemble" 
    return 0
    }
if { ([lsearch -exact $array(ENSEMBLE_MENU_LIST) $array(MR_SIMPLE_ENSEMBLE_ID,$q)] < 0) } {
    WarningMessage "Search ensemble is not in the ensemble list"
    return 0
    }
    } } 

if { (([GetValue $arrayname MR_SEARCH] == "SEARCH_OR") 
     && ( [GetValue $arrayname MR_MODE] == "AUTO") ) } {
if { $array(N_SEARCH) < 1 } {
    WarningMessage "NOT SET: search ensemble"
    return 0
} }

if { (([GetValue $arrayname MR_SEARCH] == "SEARCH_OR") 
     && ( [GetValue $arrayname MR_MODE] == "AUTO") ) } {
for { set q 1 } { $q <= $array(N_SEARCH) } { incr q } {
if { ($array(MR_FIRST_OR_ENSEMBLE_ID,$q) == "") } {
    WarningMessage "NOT SET: search ensemble" 
    return 0
    }
if { ([lsearch -exact $array(ENSEMBLE_MENU_LIST) $array(MR_FIRST_OR_ENSEMBLE_ID,$q)] < 0) } {
    WarningMessage "Search ensemble is not in the ensemble list"
    return 0
    }
} } 

if { (([GetValue $arrayname MR_ROT_SEARCH] == "SEARCH_OR")
     && ( [GetValue $arrayname MR_MODE] == "ROT") ) } {
    for { set p 1 } { $p <= $array(N_ROT_SEARCH_EXTRA) } { incr p } {
if { $array(MR_ROT_SEARCH_ID_EXTRA,$p) == "" } {
    WarningMessage "NOT SET: search ensemble" 
    return 0
    }
if { ([lsearch -exact $array(ENSEMBLE_MENU_LIST) $array(MR_ROT_SEARCH_ID_EXTRA,$p)] < 0) }  {
    WarningMessage "Search ensemble is not in the ensemble list"
    return 0
    }
} }

if { (([GetValue $arrayname MR_SEARCH] == "SEARCH_OR")
     && ( [GetValue $arrayname MR_MODE] == "AUTO") ) } {
   for { set n 1 } { $n <= $array(N_SEARCH) } { incr n } {
    for { set p 1 } { $p <= $array(N_SEARCH_EXTRA,$n) } { incr p } {
if { $array(MR_EXTRA_OR_ENSEMBLE_ID,[subst $n]_$p) == "" } {
    WarningMessage "NOT SET: search ensemble" 
    return 0
    }
if { ([lsearch -exact $array(ENSEMBLE_MENU_LIST) $array(MR_EXTRA_OR_ENSEMBLE_ID,[subst $n]_$p)] < 0) }  {
    WarningMessage "Search ensemble is not in the ensemble list"
    return 0
    }
} } }


# Check for compulsory brute force translation function AROUND search parameters
  if { ([GetValue $arrayname MR_MODE] == "TRA") 
    && ([GetValue $arrayname MR_MODE_SELECT] == "MR_BTF")
    && ([GetValue $arrayname BTF_SEARCH_OPTION] == "AROUND") } { 
    if { ($array(START_X) == "") || ($array(START_Y) == "")
      || ($array(START_Z) == "") 
      || ($array(AROUND_COORD_RANGE) == "") } {
         WarningMessage "NOT SET: translation function search (coordinates and range)" 
         return 0
    }
   }

# Check for compulsory brute force translation function REGION search parameters
  if { ([GetValue $arrayname MR_MODE] == "TRA") 
    && ([GetValue $arrayname MR_MODE_SELECT] == "MR_BTF")
    && ([GetValue $arrayname BTF_SEARCH_OPTION] == "REGION") } { 
    if { ($array(START_X) == "") || ($array(START_Y) == "")
      || ($array(START_Z) == "") || ($array(END_X) == "")
      || ($array(END_Y) == "") || ($array(END_Z) == "") } {
         WarningMessage "NOT SET: translation function search (coordinates and range)" 
         return 0
    }
   }

# Check for compulsory brute force translation function LINE search parameters
  if { ([GetValue $arrayname MR_MODE] == "TRA") 
    && ([GetValue $arrayname MR_MODE_SELECT] == "MR_BTF")
    && ([GetValue $arrayname BTF_SEARCH_OPTION] == "LINE") } { 
    if { ($array(START_X) == "") || ($array(START_Y) == "")
      || ($array(START_Z) == "") || ($array(END_X) == "")
      || ($array(END_Y) == "") || ($array(END_Z) == "") } {
         WarningMessage "NOT SET: translation function search (coordinates and range)" 
         return 0
    }
   }

  if { ( [GetValue $arrayname MR_MODE] == "TRA" ) && \
           ( $array(TRA_RLIST_FILE) == "" ) } {
         WarningMessage "NOT SET: translation function search (rlist file)" 
      return 0
  }

# Set up the array of input files: 1 or more mtz and maybe some pdb files
  set array(INPUT_FILES) "HKLIN_MAIN"

  for { set i 1 } { $i <= $array(N_ENSEMBLE) } { incr i } {
   if { [ regexp MAP [GetValue $arrayname ENSEMBLE_OPTION,$i ] ] } {
    append array(INPUT_FILES) " HKLIN,$i"
   }
   if { [ regexp PDB [GetValue $arrayname ENSEMBLE_OPTION,$i ] ] } {
   for { set j 1 } { $j <= $array(N_PDB,$i) } { incr j } {
    append array(INPUT_FILES) " XYZIN,[subst $i]_$j"
    }
  }
  }

  if { ([GetValue $arrayname SCATTERING] == "COMPONENT") } {
  for { set n 1 } { $n <= $array(N_COMPONENT) } { incr n } {
    if { $array(COMP_OPTION,$n) == "sequence file" } {
        append array(INPUT_FILES) " COMP_FILE,$n"
     }
   }
   }

  if { [GetValue $arrayname MR_MODE] == "TRA" } { 
        append array(INPUT_FILES) " TRA_RLIST_FILE"
  }
  if { ([GetValue $arrayname MR_MODE] == "LLG")
    || ([GetValue $arrayname MR_MODE] == "RNP")
    || ([GetValue $arrayname MR_MODE] == "PAK") } { 
        append array(INPUT_FILES) " SET_SOL_FILE"
  }
  if { ([GetValue $arrayname MR_MODE] != "LLG")
    && ([GetValue $arrayname MR_MODE] != "RNP")
    && ([GetValue $arrayname MR_MODE] != "PAK") 
    && ([GetValue $arrayname SET_SOL_FILE_NOTOB] != "") } { 
        append array(INPUT_FILES) " SET_SOL_FILE_NOTOB"
  }

# If a test space group other than mtz file one has been selected, then extract its space group number.
  if { ![StringSame $array(FILE_SPACEGROUP) \
              [GetValue $arrayname TEST_SPACEGROUP] ] } {
    set array(SPACGROUP_NUMBER) [GetSpaceGroupNumber $array(TEST_SPACEGROUP)]
} else {
    set array(SPACGROUP_NUMBER) ""
  }

  return 1
  }

#------------------------------------------------------------------------
 proc phaser_MR_sg { arrayname counter_sg } {
#------------------------------------------------------------------------
  upvar #0 $arrayname array
  CreateLine line \
      label "Test space group #$counter_sg" \
      message "Select a test SG (same point group)\
               or ALL to test all.\
               NB mtz SG will NOT now be tested by default."\
      widget PHASER_TEST_SG 
}

#------------------------------------------------------------------------
 proc phaser_MR_mode_update { arrayname } {
#------------------------------------------------------------------------
# Update the internal parameter MR_MODE_SELECT
# This very important parameter sets the mode that phaser will run in
  upvar #0 $arrayname array

  if { [GetValue $arrayname MR_MODE] == "AUTO" } {
     set array(MR_MODE_SELECT) "MR_AUTO"
     } elseif {
       [GetValue $arrayname MR_MODE] == "ROT" && \
       [GetValue $arrayname MR_MODE_SELECT_ROT] == "MR_FRF" } {
     set array(MR_MODE_SELECT) "MR_FRF"
     } elseif {
       [GetValue $arrayname MR_MODE] == "ROT" && \
       [GetValue $arrayname MR_MODE_SELECT_ROT] == "MR_BRF" } {
     set array(MR_MODE_SELECT) "MR_BRF"
     } elseif {
       [GetValue $arrayname MR_MODE] == "TRA" && \
       [GetValue $arrayname MR_MODE_SELECT_TRAN] == "MR_FTF" } {
      set array(MR_MODE_SELECT) "MR_FTF"
     } elseif {
       [GetValue $arrayname MR_MODE] == "TRA" && \
       [GetValue $arrayname MR_MODE_SELECT_TRAN] == "MR_BTF" } {
      set array(MR_MODE_SELECT) "MR_BTF"
     } elseif {
       [GetValue $arrayname MR_MODE] == "RNP" } {
      set array(MR_MODE_SELECT) "MR_RNP"
     } elseif {
       [GetValue $arrayname MR_MODE] == "LLG" } {
      set array(MR_MODE_SELECT) "MR_LLG"
     } elseif {
       [GetValue $arrayname MR_MODE] == "PAK" } {
      set array(MR_MODE_SELECT) "MR_PAK"
     } elseif {
       [GetValue $arrayname MR_MODE] == "ANO" } {
      set array(MR_MODE_SELECT) "MR_ANO"
     } elseif {
       [GetValue $arrayname MR_MODE] == "CCA" } {
      set array(MR_MODE_SELECT) "MR_CCA"
     } 

  }       

#------------------------------------------------------------------------
 proc phaser_MR_fractional { arrayname counter_component } {
#------------------------------------------------------------------------
# The contents of the "Composition" folder, extending frame 
  upvar #0 $arrayname array

  CreateLine line \
     message "" \
      label "Ensemble " \
      widget COMP_ENSE_ID \
      label "  Fraction scattering " \
      widget COMP_ENSE_FRAC -oblig
 }

#------------------------------------------------------------------------
 proc phaser_MR_component { arrayname counter_component } {
#------------------------------------------------------------------------
# The contents of the "Composition" folder, extending frame 
  upvar #0 $arrayname array

  CreateLine line \
      message "Select for protein or DNA/RNA as a component of the asymmetric unit" \
      label "Component #$counter_component" \
      widget PROTDNA \
      message "Enter the molecular weight of protein/nucleic acid #$counter_component" \
      label "" \
      widget COMP_OPTION -oblig \
      widget MW -oblig \
      message "Enter the number of copies of component #$counter_component in the asu " \
      label "Number in asymmetric unit" \
      widget ASYM -oblig \
     toggle_display COMP_OPTION,$counter_component open MW hide

  CreateLine line \
      message "Select for protein or DNA/RNA as a component of the asymmetric unit" \
      label "Component $counter_component" \
      widget PROTDNA \
      message "Enter the file containing sequence in single letter code or fasta format" \
      label "" \
      widget COMP_OPTION -oblig \
      message "Enter the number of copies of component $counter_component in the asu" \
      label "Number in asymmetric unit" \
      widget ASYM -oblig \
     toggle_display COMP_OPTION,$counter_component open FASTA hide

  CreateInputFileLine line \
      "Select the sequence file" "SEQ file" COMP_FILE DIR_COMP_FILE \
     -toggle_display COMP_OPTION,$counter_component open FASTA hide

  CreateLine line \
      message "Select for protein or DNA/RNA as a component of the asymmetric unit" \
      label "Component $counter_component" \
      widget PROTDNA \
      message "Enter the number of residues of protein/nucleic acid $counter_component" \
      label "" \
      widget COMP_OPTION -oblig \
      widget COMP_NRES -oblig \
      message "Enter the number of copies of component $counter_component in the asymmetric unit" \
      label "Number in asymmetric unit" \
      widget ASYM -oblig \
     toggle_display COMP_OPTION,$counter_component open NRES hide

  }

#------------------------------------------------------------------------
 proc phaser_MR_ensemble { arrayname counter_ensemble } {
#------------------------------------------------------------------------
# Sets up the main ensembles definition folder, extending frame

  upvar #0 $arrayname array

# Set a default name for ENSEMBLE_ID of ensemble1, ensemble2, ensemble3 etc
  if { $array(ENSEMBLE_ID,$counter_ensemble) == "" } {
       set array(ENSEMBLE_ID,$counter_ensemble) "ensemble$counter_ensemble" }

  CreateLine line \
      message "Enter unique identifier for ensemble #$counter_ensemble" \
      label "Ensemble Name" widget ENSEMBLE_ID \
        -command "update_phaser_MR_ensemble $arrayname" \
        -oblig \
      message "Define the ensemble by PDB file(s) or MAP file" \
      label "Define ensemble via" widget ENSEMBLE_OPTION \
      toggle_display ENSEMBLE_OPTION,$counter_ensemble open MAP

  CreateLine line \
      message "Enter unique identifier for ensemble #$counter_ensemble" \
      label "Ensemble Name" widget ENSEMBLE_ID \
        -command "update_phaser_MR_ensemble $arrayname" \
        -oblig \
      message "Define the ensemble by PDB file(s) or MAP file" \
      label "Define ensemble via" widget ENSEMBLE_OPTION \
      toggle_display ENSEMBLE_OPTION,$counter_ensemble open PDB

  OpenSubFrame frame -toggle_display ENSEMBLE_OPTION,$counter_ensemble open MAP

  CreateInputFileLine line \
      "Enter MTZ map file name (HKLIN) for ensemble # $counter_ensemble" \
      "MTZ     " \
      HKLIN DIR_HKLIN

  CreateLabinLine line \
      "Select calculated amplitude (FC) and calculated phase (PHIC)" \
      HKLIN,$counter_ensemble "FC" FC [list FC] \
      -help labin \
      -sigma "PHIC" PHIC [list PHIC]

  CreateLine line \
      message "Enter molecular weight of protein in the object\
            represented by the map (from MTZ file)" \
      label "Protein MW" widget ENSEMBLE_PROTMW -oblig \
      message "Enter molecular weight of nucleic acid in the object\
               represented by the map (from MTZ file)" \
      label "Nucleic acid MW" widget ENSEMBLE_NUCLMW -oblig \
      message "Enter the extent of the model in the X, Y and Z directions" \
      label "Extent: X" widget ENSEMBLE_EXT_X -oblig \
      label "Y" widget ENSEMBLE_EXT_Y -oblig \
      label "Z" widget ENSEMBLE_EXT_Z -oblig label "Angstroms"

  CreateLine line \
      message "Similarity to the target" \
      label "RMS error" widget ENSEMBLE_RMS_MTZ -oblig \
      message "Enter coordinates of the centre of the model, in Angstroms" \
      label "Centre: X" widget ENSEMBLE_CEN_X -oblig \
      label "Y" widget ENSEMBLE_CEN_Y -oblig \
      label "Z" widget ENSEMBLE_CEN_Z -oblig label "Angstroms" 

  CloseSubFrame

# Within the Add Ensemble toggle frame, if PDB is selected for ENSEMBLE_OPTION,
#   then insert an extending frame for adding pdb files for each ensemble

  OpenSubFrame frame -toggle_display ENSEMBLE_OPTION,$counter_ensemble open PDB 

  CreateExtendingFrame N_PDB phaser_MR_pdb \
      "Add another PDB file to the ensemble" \
      "Add superimposed PDB file to the ensemble" \
      [list XYZIN DIR_XYZIN RMS_OPTION RMS ] \
       -counter $counter_ensemble

  CloseSubFrame

# Check and update the ensemble variable menu
  edit_phaser_MR_ensemble $arrayname $counter_ensemble

  }

#------------------------------------------------------------------------
 proc phaser_MR_rot_extra { arrayname counter } {
#------------------------------------------------------------------------
# An extending frame for adding extra ensembles to a FRF search
  upvar #0 $arrayname array
  CreateLine line \
      message "Select an alternative ensemble to use in this search" \
      label "               or ensemble" \
      widget MR_ROT_SEARCH_ID_EXTRA
  }


#------------------------------------------------------------------------
 proc phaser_MR_simple_search { arrayname counter_simple_search } {
#------------------------------------------------------------------------
# The auto search extending frame code for adding extra search ensembles
  upvar #0 $arrayname array

  CreateLine line \
      message "Choose an ensemble to be used for this\
               molecular replacement search" \
      label "Perform search using " \
      widget MR_SIMPLE_ENSEMBLE_ID \
      message "The number of copies of the ensemble to look for" \
      label "  Number of copies to search for" \
      widget MR_SIMPLE_ENSEMBLE_NUM
  }


#------------------------------------------------------------------------
 proc phaser_MR_or_search { arrayname counter_search } {
#------------------------------------------------------------------------
# The auto search toggle frame code for adding extra search ensembles
  upvar #0 $arrayname array

  CreateLine line \
      label "Perform search using " \
      widget MR_FIRST_OR_ENSEMBLE_ID \
      message "The number of copies of the ensemble to look for" \
      label "  Number of copies to search for" \
      widget MR_FIRST_OR_ENSEMBLE_NUM

  CreateExtendingFrame N_SEARCH_EXTRA phaser_MR_or_search_extra \
      "Add an alternative ensemble to use in the search" \
      "Add alternative ensemble to test" \
      [list MR_EXTRA_OR_ENSEMBLE_ID N_SEARCH_EXTRA] \
      -counter $counter_search 

  }

#------------------------------------------------------------------------
 proc phaser_MR_or_search_extra { arrayname counter_search_extra counter_search } {
#------------------------------------------------------------------------
# The auto search extending frame code for adding extra ensembles to a search 
  upvar #0 $arrayname array
  CreateLine line \
      label "               or ensemble" \
      widget MR_EXTRA_OR_ENSEMBLE_ID
  }

#------------------------------------------------------------------------
 proc phaser_MR_pdb { arrayname counter_pdb counter_ensemble } {
#------------------------------------------------------------------------
# The ensemble pdb extending frame lines
  upvar #0 $arrayname array

  CreateInputFileLine line \
      "Select a PDB file associated with ensemble #$counter_ensemble " \
      "PDB #$counter_pdb" \
      XYZIN DIR_XYZIN

  CreateLine line \
      message "Specify RMS error in Angstrom, or \
               via the sequence identity (percentage or a fraction)" \
      label "Similarity of PDB #$counter_pdb to the target structure" \
      widget RMS_OPTION \
      widget RMS -oblig

  }

#------------------------------------------------------------------------
 proc update_phaser_MR_ensemble { arrayname } {
#------------------------------------------------------------------------
# This is called whenever the user clicks into the widget for an
#   ensemble name 
  upvar #0 $arrayname array
# It looks like an empty (redundant) procedure but is needed to ensure
#   that $array(N_ENSEMBLE) gets handed correctly to the menu procedure
  update_phaser_MR_ensemble_menu $arrayname $array(N_ENSEMBLE)
  }

#------------------------------------------------------------------------
 proc update_phaser_MR_ensemble_menu { arrayname n_ensemble } {
#------------------------------------------------------------------------
# Updates the variable menu of ensembles
  upvar #0 $arrayname array
# Build up list for menu
  set menu {}
  for { set n 1 } { $n <= $n_ensemble } { incr n } { lappend menu $array(ENSEMBLE_ID,$n) }
# Do the update
  UpdateVariableMenu $arrayname initialise 0 ENSEMBLE_MENU_LIST $menu 
  }

#------------------------------------------------------------------------
 proc edit_phaser_MR_ensemble { arrayname counter_ensemble } {
#------------------------------------------------------------------------
# When an ensemble is removed or added, need to update the variable menu
#   for ensemble selection.
# counter_ensemble is passed from the calling procedure. It is the 
#   number of ensembles (= number of frames)
  upvar #0 $arrayname array
  set usermenu {}
  set alias {}
# Create the ensemble menu list and its alias
  for { set n 1 } { $n <= $counter_ensemble } { incr n } { lappend usermenu "$n:  $array(ENSEMBLE_ID,$n)" }
  for { set n 1 } { $n <= $counter_ensemble } { incr n } { lappend alias "$array(ENSEMBLE_ID,$n)" }

# Update ENSEMBLE_ID if an ensemble is deleted and its ENSEMBLE_ID
#   is still set to the default ensemble$counter_ensemble
#   (which will now be out by one (one too high))
# regexp as below on "ensemble1" produces j1_whole=ensemble1
#                                         j2_root=ensemble
#                                         j3_num=1

  if { $counter_ensemble > 0 } {
     for { set p 1 } { $p <= $array(N_ENSEMBLE) } { incr p } {
         if { [regexp "^(ensemble)(\[0-9\])$" $array(ENSEMBLE_ID,$p) \
                        j1_whole j2_root j3_num] } {
            # If the trailing number (j3_num) is different from 
            #   the current molecule number (p) then reset the id
            if { $j3_num != $p } { set array(ENSEMBLE_ID,$p) "ensemble$p" }
         }
     }
  }

# Update the variable menu with the list of ensemble names
  update_phaser_MR_ensemble_menu $arrayname $counter_ensemble

  }

#------------------------------------------------------------------------
 proc update_ensemble_menu { arrayname n_ensemble } {
#------------------------------------------------------------------------
# Updates the variable menu of ensembles
  upvar #0 $arrayname array
# Build list for menu
  set menu {}
  for { set n 1} {$n <= $n_ensemble } { incr n } { lappend menu $array(ENSEMBLE_ID,$n) }
  UpdateVariableMenu $arrayname initialise 0 ENSEMBLE_MENU_LIST $menu
  }


#------------------------------------------------------------------------
 proc phaser_MR_set_test_space_group { arrayname } {
#------------------------------------------------------------------------
# To set up the choice of test space groups possible (those in the same
#   point group) given the space group from the mtz file
  upvar #0 $arrayname array
  set spgp_list {}
  set spgp_alias {}
  set laue_no [GetLaueGroupNumber $array(FILE_SPACEGROUP)]
# laue_no is zero if FILE_SPACEGROUP is not in the standard list
# update the list of spacegroups varlabel
  phaser_MR_updatevarlabel $arrayname FILE_SPACEGROUP
  if { $laue_no <= 0 } {
    lappend spgp_list $array(FILE_SPACEGROUP)
    lappend spgp_alias [GetSpaceGroupNumber $array(FILE_SPACEGROUP)]
  } else {
    set spgp_alias [GetLaueGroup $laue_no]
    foreach gp_no $spgp_alias {
      lappend spgp_list [GetSpaceGroupCode $gp_no]
    }
  }
# GetLaueGroup returns a list of space groups for the given Laue Group

  UpdateVariableMenu $arrayname initialise [llength $array(LAUE_SPGP_LIST)] \
        LAUE_SPGP_LIST $spgp_list \
        MODEL_ALIAS_LIST $spgp_alias

  set array(PHASER_TEST_SG) [GetSpaceGroupCode $array(FILE_SPACEGROUP)]

}

#------------------------------------------------------------------------
 proc phaser_MR_updatevarlabel { arrayname FILE_SPACEGROUP } {
#------------------------------------------------------------------------
# procedure to update the FILE_SPACEGROUP varlabel
  upvar #0 $arrayname array
  set field_list [GetWidget $arrayname FILE_SPACEGROUP]
  foreach field $field_list {
    if { [winfo exists $field] } { $field configure -text $array(FILE_SPACEGROUP) } }
  }

#------------------------------------------------------------------------
  proc phaser_MR_tog_activate {arrayname} {
#------------------------------------------------------------------------
  upvar #0 $arrayname array
  if { [StringSame $array(MR_MODE_SELECT) MR_FRF] } { 
    set array(TOG_FRF_TARGET) "1"
  } elseif { [StringSame $array(MR_MODE_SELECT) MR_FTF] } {
    set array(TOG_FTF_TARGET) "1"
  }

  }

#------------------------------------------------------------------------
  proc phaser_MR_set_reso {arrayname} {
#------------------------------------------------------------------------
  upvar #0 $arrayname array
  if { ( $array(RESOLUTION_DMIN) >= 2.5 ) } {
     set array(RESOLUTION_DMIN_MR) [GetValue $arrayname RESOLUTION_DMIN]
     } else {
     set array(RESOLUTION_DMIN_MR) 2.5
     }
}

#------------------------------------------------------------------------
  proc phaser_MR_maca { arrayname counter_s } {
#------------------------------------------------------------------------
  global configure
  upvar #0 $arrayname array
  CreateLine line \
      label "Refine: Anisotropic" \
      widget MACA_ANIS \
      label "Bins" \
      widget MACA_BINS \
      label "SolK" \
      widget MACA_SOLK \
      label "SolB" \
      widget MACA_SOLB \
      label "Number of Cycles" \
      widget MACA_NCYC 
  }

#------------------------------------------------------------------------
  proc phaser_MR_macm { arrayname counter_s } {
#------------------------------------------------------------------------
  global configure
  upvar #0 $arrayname array
  CreateLine line \
      label "Refine: Orientation" \
      widget MACM_ROT \
      label "Translation" \
      widget MACM_TRA \
      label "B-factor" \
      widget MACM_BFAC \
      label "V-rms" \
      widget MACM_VRMS \
      label "Number of Cycles" \
      widget MACM_NCYC 
  }

#------------------------------------------------------------------------
  proc phaser_MR_task_window {arrayname} {
#------------------------------------------------------------------------

  upvar #0 $arrayname array
  global configure

  if { [CreateTaskWindow $arrayname \
        "Maximum Likelihood Molecular Replacement"\
        "Phaser" \
        [ list "Define data" \
               "Define ensembles (models)" \
               "Define composition of the asymmetric unit" \
               "Search parameters " \
               "Additional parameters" \
               "Output control" \
               "Expert parameters" ] \
       ] == 0 } return

# SetProgramHelpFile phaser


#=PROTOCOL===============================================================

  OpenFolder protocol

  CreateTitleLine line TITLE

# Sets up the central menus for choosing the mode of phaser
#   whether rotation, translation or refinement/rescoring searches

  CreateLine line \
      label "Mode for molecular replacement" \
      message "The central menu: select the molecular replacement\
               mode for phaser to run" \
      widget MR_MODE \
        -command "phaser_MR_mode_update $arrayname" \
      toggle_display MR_MODE open [list ANO RNP LLG PAK AUTO CCA ]

  CreateLine line \
      label "Mode for molecular replacement" \
      message "The central menu: select the molecular replacement\
               mode for phaser to run" \
      widget MR_MODE \
        -command "phaser_MR_mode_update $arrayname" \
      message "RF: rotation function;\
               ML: maximum likelihood" \
      label "using" \
      widget MR_MODE_SELECT_ROT \
        -command "phaser_MR_mode_update $arrayname" \
      toggle_display MR_MODE open ROT

  CreateLine line \
      label "Mode for molecular replacement" \
      message "The central menu: select the molecular replacement\
               mode for phaser to run" \
      widget MR_MODE \
        -command "phaser_MR_mode_update $arrayname" \
      message "TF: translation function;\
               ML: maximum likelihood" \
      label "using" \
      widget MR_MODE_SELECT_TRAN \
        -command "phaser_MR_mode_update $arrayname" \
      toggle_display MR_MODE open TRA

  CreateLine line \
      message "Number of processors (parallelization)" \
      widget TOG_NJOBS -toggleon \
      label "Number of processors "\
      widget NJOBS \
      label "(only relevant if phaser compiled with openmp option)"

#=FILE===================================================================

# The "Define data" folder

  OpenFolder 1 MR_MODE 

  CreateInputFileLine line \
      "Enter MTZ file name (HKLIN)" \
      "MTZ in     " \
      HKLIN_MAIN DIR_HKLIN_MAIN \
        -setfileparam resolution_min RESOLUTION_DMAX \
        -setfileparam resolution_max RESOLUTION_DMIN \
        -setfileparam resolution_min RESOLUTION_AUTO_DMAX \
        -setfileparam resolution_max RESOLUTION_AUTO_DMIN \
        -setfileparam cell_1 CELL_1 \
        -setfileparam cell_2 CELL_2 \
        -setfileparam cell_3 CELL_3 \
        -setfileparam cell_4 CELL_4 \
        -setfileparam cell_5 CELL_5 \
        -setfileparam cell_6 CELL_6 \
        -setfileparam space_group_name FILE_SPACEGROUP \
        -command "phaser_MR_set_test_space_group $arrayname" \
        -command "phaser_MR_set_reso $arrayname"

   CreateLabinLine line \
       "Select amplitude (F) and sigma (SIGF)" \
       HKLIN_MAIN "F" F {NULL} \
       -sigma "SIGF" SIGF {NULL}

  CreateLine line \
      message "Default is to take all data from low resolution limit\
               to the greater of dmin and 2.5 A" \
      widget TOG_RESOLUTION -toggleon \
      label "Resolution range" \
      message "Enter dmax, in Angstroms" \
      widget RESOLUTION_DMAX \
      label "A to" \
      message "If the dataset goes to higher than 2.5 A resolution,\
               then a default dmin of 2.5 A is set." \
      widget RESOLUTION_DMIN_MR \
      label "A" \
      toggle_display MR_MODE hide [list CCA PAK]

  CreateLine line \
      message "Space group read from mtz file.\
               By default, phaser is run with this space group" \
      label "Space group read from mtz file" \
      varlabel FILE_SPACEGROUP \
      label "; " \
      widget TOG_SPACEGROUP -toggleon \
      message "Run Phaser with a different space group" \
      label "Run Phaser with space group" \
      widget TEST_SPACEGROUP \
      toggle_display MR_MODE open [list ROT RNP LLG PAK]

  OpenSubFrame frame -toggle_display MR_MODE open [list AUTO TRA]

  CreateLine line \
      message "Space group read from mtz file." \
      label "Space group read from mtz file" \
      varlabel FILE_SPACEGROUP \
      label "; " \
      message "Specify the space group(s) to run Phaser with" \
      label "Run Phaser with" \
      widget SPACEGROUP_OPTION 

  CreateLine line \
      widget TOG_SGAL_SORT -toggleon \
      label "Test spacegroups in order of general occurrence in PDB" \
      widget SGAL_SORT_ONOFF \
      toggle_display SPACEGROUP_OPTION open [list SG_ALT SG_ALL] 

  OpenSubFrame frame -toggle_display SPACEGROUP_OPTION open SG_ALT

  CreateExtendingFrame N_SG phaser_MR_sg \
       "Add another test space group" "Add another space group" \
       [list PHASER_TEST_SG ] 

  CloseSubFrame

  CloseSubFrame

# $line.l2 configure -width 7 -background $configure(COLOUR_BACKGROUND)

#========================================================================
# The "Define ensembles" folder

  OpenFolder 2 MR_MODE hide [list ANO CCA ] open

  CreateToggleFrame N_ENSEMBLE phaser_MR_ensemble \
      "Define another ensemble" \
      "Ensemble #" \
      "Add ensemble" \
      [list ENSEMBLE_ID ENSEMBLE_OPTION HKLIN DIR_HKLIN ENSEMBLE_LABIN FC PHIC N_PDB ] \
       -edit_proc edit_phaser_MR_ensemble \
       -child phaser_MR_pdb

#========================================================================
# The "Define composition of the asymmetric unit" folder

  OpenFolder 3 MR_MODE hide [list PAK ] open 

  CreateLine line \
      message "Total scattering of atoms in the asymmetric unit" \
      label "Total scattering determined by "\
      widget SCATTERING \
      toggle_display SCATTERING hide SOLVENT 

  CreateLine line \
      message "Total scattering of atoms in the asymmetric unit" \
      label "Total scattering determined by "\
      widget SCATTERING \
      widget SCATTERING_SOLVENT  -oblig \
      toggle_display SCATTERING open SOLVENT 

  OpenSubFrame frame -toggle_display SCATTERING open COMPONENT hide

  CreateExtendingFrame N_COMPONENT phaser_MR_component \
      "Define another protein or nucleic acid component" "Define another component" \
      [list PROTDNA MW ASYM COMP_OPTION COMP_NRES COMP_FILE DIR_COMP_FILE ]

  CloseSubFrame

  OpenSubFrame frame -toggle_display SCATTERING open FRAC hide

  CreateExtendingFrame N_COMPONENT_FRAC phaser_MR_fractional \
      "Define another protein or nucleic acid component" "Define another component" \
      [list COMP_ENSE_ID COMP_ENSE_FRAC ]

  CloseSubFrame

#========================================================================
# The "Search parameters" folder 

  OpenFolder 4 MR_MODE hide [list CCA ANO] open

  CreateLine line \
      message "A SOL file is a Phaser search results file" \
      label "Define known solutions from a SOL file" \
     toggle_display MR_MODE open [list LLG RNP PAK] hide

  CreateInputFileLine line \
      "Define known solutions from a SOL file \
       (Phaser search results file)" \
      "SOL file" \
      SET_SOL_FILE DIR_SET_SOL_FILE \
     -toggle_display MR_MODE open [list LLG RNP PAK] hide

  CreateLine line \
      message "A .rlist file is a Phaser rotation search results file" \
      label "Define orientations for translation search\
             and any known solutions from a .rlist file" \
     toggle_display MR_MODE open [list TRA ] hide


  CreateInputFileLine line \
      "A .rlist file is a Phaser rotation search results file" \
      ".rlist file" \
      TRA_RLIST_FILE DIR_TRA_RLIST_FILE \
     -toggle_display MR_MODE open [list TRA ] hide

  OpenSubFrame frame -toggle_display MR_MODE_SELECT open MR_BTF hide

  CreateLine line \
      message "Select search type: full, around a point,\
              within a region, along a line or within a plane" \
      label "Perform" \
      widget BTF_SEARCH_OPTION \
      label "translation search" 

  CreateLine line \
      message "Select the coordinate system" \
      label "Search around" \
      widget COORD_SYS \
      message "Enter the search coordinates" \
      label "coordinates X" \
      widget START_X -oblig \
      label "Y" \
      widget START_Y -oblig \
      label "Z" \
      widget START_Z -oblig \
      message "Enter in Angstroms the range around XYZ to search" \
      label "with search range" \
      widget AROUND_COORD_RANGE -oblig \
      toggle_display BTF_SEARCH_OPTION open AROUND

  CreateLine line \
      message "Select the coordinate system the region is defined in" \
      label "Region:" \
      widget COORD_SYS \
      message "Enter the start coordinates X,Y,Z" \
      label "coords X" \
      widget START_X -oblig \
      label "Y" \
      widget START_Y -oblig \
      label "Z" \
      widget START_Z -oblig \
      message "Enter the end coordinates X,Y,Z" \
      label "and X" \
      widget END_X -oblig \
      label "Y" \
      widget END_Y -oblig \
      label "Z" \
      widget END_Z -oblig \
      toggle_display BTF_SEARCH_OPTION open [list REGION LINE] hide

  CloseSubFrame

  OpenSubFrame frame -toggle_display MR_MODE_SELECT open MR_BRF hide

  CreateLine line \
      message "Select to perform a full or restricted search" \
      label "Search" \
      widget BRF_SEARCH_OPTION \
      message "Restrict the search to region around this set of Euler angles" \
      label "Alpha" \
      widget AROUND_ALPHA -oblig \
      label "Beta" \
      widget AROUND_BETA -oblig \
      label "Gamma" \
      widget AROUND_GAMMA -oblig \
      message "Rotational distance around the given Euler angle set" \
      label "Range" \
      widget AROUND_EULER_RANGE -oblig \
      toggle_display BRF_SEARCH_OPTION open AROUND 

  CreateLine line \
      message "Select to perform a full or restricted search" \
      label "Search" \
      widget BRF_SEARCH_OPTION \
      toggle_display BRF_SEARCH_OPTION open FULL 

  CloseSubFrame

  OpenSubFrame frame -toggle_display MR_MODE open [list ROT] hide

  CreateLine line \
       label "Allow search with alternative ensembles (models) for a single component of the ASU" \
       message "Set toggle on for OR search" \
       widget MR_ROT_SEARCH 

  CreateLine line \
      message "Choose an ensemble to be used for the rotation search" \
      label "Perform search using " \
      widget MR_ROT_SEARCH_ID

  OpenSubFrame frame -toggle_display MR_ROT_SEARCH open SEARCH_OR hide

  CreateExtendingFrame N_ROT_SEARCH_EXTRA phaser_MR_rot_extra \
      "Add an alternative ensemble to be searched for" \
      "Add an alternative search ensemble" \
      [list MR_ROT_SEARCH_ID_EXTRA]

  CloseSubFrame

  CloseSubFrame

  OpenSubFrame frame -toggle_display MR_MODE open AUTO hide

  CreateLine line \
     message "Use FAST if you expect the TFZ to be high for each copy" \
     widget TOG_SEARCH_METHOD -toggleon \
     label "Search method for multiple copies in asymmetric unit " \
     widget SEARCH_METHOD \
     toggle_display SEARCH_METHOD open FULL hide

  CreateLine line \
     message "Use FAST if you expect the TFZ to be high for each copy" \
     widget TOG_SEARCH_METHOD -toggleon \
     label "Search method for multiple copies in asymmetric unit " \
     widget SEARCH_METHOD \
     label "and search down percent" \
     message "Value for reducing RF peak cutoff on 2nd pass (more peaks)" \
     widget SEARCH_DOWN_PERC \
     toggle_display SEARCH_METHOD open FAST hide

  CreateLine line \
      widget TOG_SEARCH_ORDER_AUTO -toggleon \
      label "Determine ensemble search order automatically" \
      widget SEARCH_ORDER_AUTO_ONOFF 

  CreateLine line \
       label "Allow search with alternative ensembles (models) for a single component of the ASU" \
       message "Set toggle on for OR search" \
       widget MR_SEARCH

  OpenSubFrame frame -toggle_display MR_SEARCH open SEARCH_SIMPLE hide

  CreateExtendingFrame N_SIMPLE_SEARCH phaser_MR_simple_search \
      "Add another ensemble to be searched for" \
      "Add another search" \
      [list MR_SIMPLE_ENSEMBLE_ID MR_SIMPLE_ENSEMBLE_NUM ]

  CloseSubFrame

  OpenSubFrame frame -toggle_display MR_SEARCH open SEARCH_OR hide

  CreateToggleFrame N_SEARCH phaser_MR_or_search \
      "Perform another mr auto search" \
      "Search #" \
      "Add another search" \
      [list MR_FIRST_OR_ENSEMBLE_ID MR_FIRST_OR_ENSEMBLE_NUM MR_EXTRA_OR_ENSEMBLE_ID N_SEARCH_EXTRA ] \
      -child phaser_MR_or_search_extra 

  CloseSubFrame

  CloseSubFrame

#========================================================================
# The "Additional parameters" folder

  OpenFolder 5 MR_MODE  closed

  CreateLine line \
      label "No additional parameters" -italic \
     toggle_display MR_MODE open [list RNP LLG CCA ANO] hide

  CreateLine line \
      message "A SOL file is a Phaser search results file" \
      label "Define any known solutions from a SOL file" \
     toggle_display MR_MODE open [list AUTO ROT] hide

  CreateInputFileLine line \
      "Define known solutions from a SOL file\
       (Phaser search results file)" \
      "SOL file" \
      SET_SOL_FILE_NOTOB DIR_SET_SOL_FILE_NOTOB \
     -notoblig \
     -toggle_display MR_MODE open [list AUTO ROT] hide

  OpenSubFrame frame -toggle_display MR_MODE open [list AUTO PAK] hide

  CreateLine line \
       message "Solutions with the lowest number of clashes will be saved \
               if less than the max allowed (default 10)" \
      widget TOG_PACK -toggleon \
      label "Packing criterion " \
      widget PACK_OPTION \
      label "Allow " \
      widget PACK_ALLOWED_CLASHES \
      toggle_display PACK_OPTION open BEST

  CreateLine line \
       message "All solutions will be saved no matter how many clashes" \
      widget TOG_PACK -toggleon \
      label "Packing criterion " \
      widget PACK_OPTION \
      toggle_display PACK_OPTION open ALL

  CreateLine line \
       message "Default for allowed C-alpha clashes is 10;\
                increase for low-homology models" \
      widget TOG_PACK -toggleon \
      label "Packing criterion " \
      widget PACK_OPTION \
      label "Allow " \
      widget PACK_ALLOWED_CLASHES \
      toggle_display PACK_OPTION open ALLOWED

  CreateLine line \
       message "Default for allowed C-alpha clashes is 10;\
                increase for low-homology models" \
      widget TOG_PACK -toggleon \
      label "Packing criterion " \
      widget PACK_OPTION \
      label "Allow " \
      widget PACK_ALLOWED_CLASHES \
      toggle_display PACK_OPTION open PERCENT

  CreateLine line \
      widget TOG_PACK_QUICK -toggleon \
      label "Quick Packing " \
      widget PACK_QUICK_ONOFF \
      toggle_display PACK_OPTION hide ALL

  CloseSubFrame

  OpenSubFrame frame -toggle_display MR_MODE open [list AUTO ROT] hide

  CreateLine line \
      message "Final selection criterion for peaks from rotation searches" \
      widget TOG_FINAL_ROT -toggleon \
      label "Rotation search peak selection" \
      message "Default 75 percent" \
      widget FINAL_OPTION_ROT \
      message "Enter the cutoff" \
      widget FINAL_SIGMA_ROT \
      toggle_display FINAL_OPTION_ROT hide [list ALL] open 

  CreateLine line \
      message "Final selection criterion for peaks from rotation searches" \
      widget TOG_FINAL_ROT -toggleon \
      label "Rotation search peak selection" \
      message "Default 75 percent" \
      widget FINAL_OPTION_ROT \
      toggle_display FINAL_OPTION_ROT open [list ALL] hide 

  CreateLine line \
      message "Select whether clustered or raw peaks are saved" \
      widget TOG_PEAKS_ROT_CLUSTER -toggleon \
      label "Rotation clustering" \
      widget PEAKS_ROT_CLUSTER

  CloseSubFrame

  OpenSubFrame frame -toggle_display MR_MODE open [list AUTO TRA] hide

  CreateLine line \
      message "Final selection criterion for peaks from translation searches" \
      widget TOG_FINAL_TRA -toggleon \
      label "Translation search peak selection" \
      message "Default 75 percent" \
      widget FINAL_OPTION_TRA \
      message "Enter the cutoff" \
      widget FINAL_SIGMA_TRA \
      toggle_display FINAL_OPTION_TRA hide [list ALL] open 

  CreateLine line \
      message "Final selection criterion for peaks from translation searches" \
      widget TOG_FINAL_TRA -toggleon \
      label "Translation search peak selection" \
      message "Default 75 percent" \
      widget FINAL_OPTION_TRA \
      toggle_display FINAL_OPTION_TRA open [list ALL] hide 

  CreateLine line \
      message "Select whether clustered or raw peaks are to be rescored" \
      widget TOG_PEAKS_TRA_CLUSTER -toggleon \
      label "Translation clustering" \
      widget PEAKS_TRA_CLUSTER

  CloseSubFrame

  OpenSubFrame frame -toggle_display MR_MODE open [list AUTO ]

  CreateLine line \
      message "Choose whether to purge the solutions from TF and RNP \
               (when searching multiple space groups)" \
      widget TOG_PURGE -toggleon \
      label "Purge solutions" \
      widget PURGE_ONOFF \
      label "percent of top peak" \
      widget PURGE_PERCENT \
      toggle_display PURGE_ONOFF open ON

  CreateLine line \
      message "Choose whether to purge the solutions from TF and RNP \
               (when searching multiple space groups)" \
      widget TOG_PURGE -toggleon \
      label "Purge solutions" \
      widget PURGE_ONOFF \
      toggle_display PURGE_ONOFF open OFF

  CreateLine line \
      message "Choose whether to permute the order of the search set.\
               Default is OFF." \
      widget TOG_PERMUTATIONS -toggleon \
      label "Permute search set" \
      widget PERMUTATIONS_ONOFF 

  CreateLine line \
      message "Default is to take all data in mtz file" \
      widget TOG_RESOLUTION_AUTO -toggleon \
      label "Resolution range for high resolution refinement" \
      message "Enter dmax, in Angstroms" \
      widget RESOLUTION_AUTO_DMAX \
      label "A to" \
      widget RESOLUTION_AUTO_DMIN \
      label "A" 

  CloseSubFrame

#========================================================================
# The "Output control" folder

  OpenFolder 6 MR_MODE closed

  CreateLine line \
      message "VERBOSE: more information included in log file than by default" \
      widget TOG_VERBOSE \
      label "Verbose output to log file" \

  CreateLine line \
      message "DEBUG: debugging information included in log file than by default" \
      widget TOG_VERBOSE_EXTRA \
      label "Debug output to log file" 

  CreateLine line \
      message "Default number of top pdbfiles and mtzfiles is 1" \
      widget TOG_TOPF -toggleon \
      label "Number of top solutions to output as PDB and MTZ files"\
      widget XYZOUT_NPDB \
      toggle_display MR_MODE hide [list ANO CCA]

  CreateLine line \
      message "Output the results in a .sol file" \
      widget TOG_KEYWDS -toggleon \
      label "SOL file output" \
      widget KEYWDS_ONOFF \
      toggle_display MR_MODE hide [list ROT]

  CreateLine line \
      message "Output the results in a .rlist file" \
      widget TOG_KEYWDS -toggleon \
      label "RLIST file output" \
      widget KEYWDS_ONOFF \
      toggle_display MR_MODE open [list ROT] hide

  CreateLine line \
      message "Controls output of PDB files containing potential solutions.\
               Default depends on mode." \
      widget TOG_XYZOUT -toggleon \
      label "PDB file output" \
      widget XYZOUT_ONOFF \
      toggle_display MR_MODE hide [list ANO CCA]

  CreateLine line \
      message "Controls output of MTZ files with phase information from solutions.\
               Default depends on mode." \
      widget TOG_HKLOUT -toggleon \
      label "MTZ file output" \
      widget HKLOUT_ONOFF \
      toggle_display MR_MODE hide [list CCA PAK ]

#========================================================================
# The "Expert parameters" folder

  OpenFolder 7 MR_MODE hide [list CCA ] closed

  CreateLine line \
      message "Select whether or not to rescore fast search solutions" \
      widget TOG_RESCORE_ROT -toggleon \
      label "Rescoring rotation" \
      widget RESCORE_ONOFF_ROT \
      -command "phaser_MR_rescore_parameter $arrayname" \
      toggle_display MR_MODE_SELECT open [list MR_AUTO MR_FRF] hide

  OpenSubFrame frame -toggle_display MR_MODE_SELECT open [list MR_AUTO MR_FTF] hide

  CreateLine line \
      message "Select whether or not to rescore fast search solutions" \
      widget TOG_RESCORE_TRA -toggleon \
      label "Rescoring translation" \
      widget RESCORE_ONOFF_TRA \
      -command "phaser_MR_rescore_parameter $arrayname" \
      toggle_display SEARCH_METHOD open [list FULL] hide

  CreateLine line \
      message "Select whether or not to rescore fast search solutions" \
      widget TOG_RESCORE_FAST -toggleon \
      label "Rescoring translation" \
      widget RESCORE_ONOFF_FAST \
      -command "phaser_MR_rescore_parameter $arrayname" \
      label "(fast search method)" \
      toggle_display SEARCH_METHOD open [list FAST] hide

  CloseSubFrame

  CreateLine line \
      message "BINS: specify how the binning to all reflections will apply"\
      widget TOG_BINS -toggleon \
      label "Bins: Min" widget BINS_L label "Max" widget BINS_H label "Width" widget BINS_W \
      message "Binning function: AS^3 + BS^2 + CS\
               where S=1/reso.\
               A,B,C must all be +ve and A=B=0 or A=0 or B=0"\
      label "Cubic: A" widget BINS_A label "B" widget BINS_B label "C" widget BINS_C \
      toggle_display MR_MODE hide [list PAK CCA ]

  CreateLine line \
      message "BOXSCALE: specify how finely the ensemble structure factors\
               are sampled in reciprocal space" \
      widget TOG_BOXSCALE -toggleon \
      label "Box scale factor" \
      message "The ensemble 'cell' is <BOXSCALE>*(extent of molecule)" \
      widget BOXSCALE \
      toggle_display MR_MODE hide [list ANO CCA PAK]

  CreateLine line \
    widget TOG_CELL -toggleon \
    label "Unit Cell: a" \
    message "Cell dimensions (default from MTZ file) (CELL)" \
    widget CELL_1 -width 8 \
    label "b" \
    widget CELL_2 -width 8 \
    label "c" \
    widget CELL_3 -width 8 \
    label "alpha" \
    widget CELL_4 -width 8 \
    label "beta" \
    widget CELL_5 -width 8 \
    label "gamma" \
    widget CELL_6 -width 8

  CreateLine line \
      message "CLMN: radius for decomposition of the Patterson" \
      widget TOG_CLMN -toggleon \
      label "CLMN sphere outer radius " \
      message "Specify outer radius in Angstroms. Min acceptable value is 5 A" \
      widget CLMN_SPHERE \
      label "LMIN " \
      widget CLMN_INNER \
      label "LMAX " \
      widget CLMN_OUTER \
      toggle_display MR_MODE_SELECT open [list MR_AUTO MR_FRF ] hide

  OpenSubFrame frame -toggle_display MR_MODE_SELECT hide [list CCA MR_PAK] open

    CreateLine line \
      message "Refinement macrocyles" \
      widget TOG_ANISO -toggleon \
      label "Anisotropic refinement protocol" \
      widget MACA_PROTOCOL

    OpenSubFrame frame \
      -toggle_display MACA_PROTOCOL open CUSTOM hide
     CreateExtendingFrame N_MACA phaser_MR_maca \
        "Customize the anisotropic refinement" \
        "Add another anisotropic macrocycle" \
        [list MACA_ANIS MACA_BINS MACA_SOLK MACA_SOLB MACA_NCYC ]
    CloseSubFrame

  CloseSubFrame

  OpenSubFrame frame -toggle_display MR_MODE_SELECT open [list MR_AUTO MR_RNP] hide

    CreateLine line \
      message "Refinement macrocyles" \
      widget TOG_MACM -toggleon \
      label "Molecular replacement refinement protocol" \
      widget MACM_PROTOCOL

    OpenSubFrame frame \
      -toggle_display MACM_PROTOCOL open CUSTOM hide
     CreateExtendingFrame N_MACM phaser_MR_macm \
        "Customize the MR refinement" \
        "Add another MR macrocycle" \
        [list MACM_ROT MACM_TRA MACM_BFAC MACM_VRMS MACM_NCYC ]
    CloseSubFrame

  CloseSubFrame

  OpenSubFrame frame -toggle_display MR_MODE hide [list ANO CCA PAK] open

  CreateLine line \
      message "OUTLIER: choose whether to reject large, unlikely E-values" \
      widget TOG_OUTLIER -toggleon \
      label "Outlier rejection" \
      widget OUTLIER_OPTION \
      label "outlier probability" \
      message "Enter cutoff probability, where outliers with a probability\
               less than the cutoff are rejected" \
      widget OUTLIER_PROB -width 10 \
      toggle_display OUTLIER_OPTION open ON 

  CreateLine line \
      message "OUTLIER: choose whether to reject large, unlikely E-values.\
               Default is ON with probability 0.00001" \
      widget TOG_OUTLIER -toggleon \
      label "Outlier rejection" \
      widget OUTLIER_OPTION \
      toggle_display OUTLIER_OPTION open OFF

  CloseSubFrame

  CreateLine line \
      message "PACKING: sampling of search space" \
      widget TOG_PACK_DISTANCE -toggleon \
      label "Distance for the packing test" \
      message "Distance between marker atoms for the clash test " \
      widget PACK_DISTANCE \
      toggle_display MR_MODE_SELECT open [list MR_AUTO MR_PAK]

  CreateLine line \
      message "PTGROUP: tolerances for detection of model symmetry" \
      widget TOG_PTGR_SEQ -toggleon \
      label "Point Group Minimum Sequence Tolerances: Coverage" \
      widget PTGR_COVERAGE  \
      label "Identity" \
      widget PTGR_IDENTITY \
      toggle_display MR_MODE_SELECT open [list MR_AUTO MR_RNP]

  CreateLine line \
      message "PTGROUP: tolerances for detection of model symmetry" \
      widget TOG_PTGR_XYZ -toggleon \
      label "Point Group Maximum Coordinate Tolerances: RMS" \
      widget PTGR_RMSD -width 6 \
      label "Angular" \
      widget PTGR_TOL_ANG -width 6 \
      label "Spatial" \
      widget PTGR_TOL_SPC -width 6 \
      toggle_display MR_MODE_SELECT open [list MR_AUTO MR_RNP]

  CreateLine line \
      message "SAMPLING: sampling of search space" \
      widget TOG_SAMP_ROT -toggleon \
      label "Sampling for rotation search" \
      message "Sampling for rotation search given in degrees" \
      widget SAMPLING_ROT \
      toggle_display MR_MODE open [list AUTO ROT]

  CreateLine line \
      message "SAMPLING: sampling of search space" \
      widget TOG_SAMP_TRA -toggleon \
      label "Sampling for translation search" \
      message "Sampling for translation search given in Angstroms" \
      widget SAMPLING_TRA \
      toggle_display MR_MODE open [list AUTO TRA] 

  CreateLine line \
      message "RESHARPEN: B-factor added back to map coefficients" \
      widget TOG_RESHARPEN -toggleon \
      label "Resharpening percentage " \
      message "Resharpening percentage " \
      widget RESHARPEN_PERC \
      toggle_display MR_MODE hide [list CCA PAK] 

  CreateLine line \
      message "SGALTERNATIVE Sort alternative space groups by frequency of occurrance" \
      widget TOG_SGAL_SORT -toggleon \
      label "Sort alternative spacegroups " \
      widget SGAL_SORT_ONOFF \
      toggle_display MR_MODE hide [list ANO CCA PAK ROT TRA]

  CreateLine line \
      message "SOLPARAMETERS: set the solvent parameters for the ensembling" \
      widget TOG_SOLP -toggleon \
      label "Solvent Parameters: FSOL " \
      widget SOLP_FSOL -toggleon \
      label " BSOL " \
      widget SOLP_BSOL \
      toggle_display MR_MODE hide [list ANO CCA PAK]

  CreateLine line \
      message "TARGET: select the rotation target function to be used" \
      widget TOG_FRF_TARGET -toggleon \
      label "Rotation target function" \
      widget FRF_TARGET_OPTION \
      -command "phaser_MR_tog_activate $arrayname" \
      toggle_display MR_MODE_SELECT open [list MR_FRF MR_AUTO] hide

  CreateLine line \
      message "TARGET: select the translation target function to be used" \
      widget TOG_FTF_TARGET -toggleon \
      label "Translation target function" \
      widget FTF_TARGET_OPTION \
      -command "phaser_MR_tog_activate $arrayname" \
      toggle_display MR_MODE_SELECT open [list MR_FTF MR_AUTO] hide

  CreateLine line \
      message "ZSCORE: TFZ for definite solution in FAST search mode" \
      widget TOG_ZSCORE -toggleon \
      label "TFZ for amalgamating solutions: maximum of" \
      widget ZSCORE_SOLVED \
      label "and half highest TFZ" \
      widget ZSCORE_HALF \
      toggle_display MR_MODE_SELECT open [list MR_AUTO] hide

  }

#use puts to output debugging info to screen
