# =======================================================================
# phaser_NMA.tcl --
#
# Normal Mode Analysis
#
# CCP4Interface
#
# =======================================================================

#------------------------------------------------------------------------------
proc phaser_NMA_prereq { } {
#------------------------------------------------------------------------------
  if { ![file exists [FindExecutable phaser]] } {
    return 0
  }
  return 1
}

#------------------------------------------------------------------------
  proc phaser_NMA_setup { typedefVar arrayname } {
#------------------------------------------------------------------------
  upvar #0 $typedefVar typedef
  upvar #0 $arrayname array

  DefineMenu _eigen_type \
    [list "Write to file" "Do not write to file" "Read from file"] \
    [list "WRITEON" "WRITEOFF" "READ"] 

  DefineMenu _perturb_type \
    [list "  RMS  " "  DQ   " ] [list "RMS" "DQ"]

  DefineMenu _method_type \
    [list "rotation-translation blocks" "C-alpha atoms only" "All atoms" ] \
    [list "RTB" "CA" "ALL" ]

  DefineMenu _perturb_dir \
    [list "forward" "backward" "to-and-fro" ] [list "FORWARD" "BACKWARD" "TOFRO"]

  set typedef(_onoff) \
    { menu { "on" "off" } { ON OFF } }

  DefineMenu _rmsid_option \
   [list "rms difference" "sequence identity" ] [list "RMS" "IDENT" ]

  set typedef(_mat_file)        { file MATRIX ".mat" "Phaser matrix file" { "Text Browser" } { DisplayTextFile } }

  return 1
  }

#------------------------------------------------------------------------
proc phaser_NMA_run { arrayname } {
#------------------------------------------------------------------------
  upvar #0 $arrayname array

  set array(OUTPUT_FILES) ""

  if { $array(NMA_XYZIN) == "" } {
     WarningMessage "NOT SET: pdb file"
     return 0
  }
  if { $array(ENSEMBLE_ID_NMA) == "" } {
     WarningMessage "NOT SET: ensemble name for SOL file output"
     return 0
  }
  if { $array(RMS_NMA) == "" } {
     WarningMessage "NOT SET: sequence identity or RMS deviation for SOL file output"
     return 0
  }
  if { ([GetValue $arrayname NMA_EIGEN] == "READ") && $array(EIGENIN) == "" } {
     WarningMessage "NOT SET: eigen file"
     return 0
  }
  if { $array(N_NMA_MODES) == 0 } {
     WarningMessage "NOT SET: modes for perturbation"
     return 0
  }
  for { set n 1 } { $n <= $array(N_NMA_MODES) } { incr n } {
    if { $array(NMA_MODE,$n) == "" } {
     WarningMessage "NOT SET: normal mode perturbation value"
     return 0
    }
    if { $array(NMA_MODE,$n) < 7 } {
     WarningMessage "Normal modes less than 7 are trival modes"
     return 0
    }
  }
  if { ([GetValue $arrayname NMA_METHOD] == "RTB") } {
  if { $array(TOG_NMA_MAXB) && $array(NMA_MAXB) == "" } {
     WarningMessage "NOT SET: Maximum number of blocks"
     return 0
  }
  if { $array(TOG_NMA_NRES) && $array(NMA_NRES) == "" } {
     WarningMessage "NOT SET: Number of residues per block"
     return 0
  }
  }
  if { ([GetValue $arrayname NMA_PERTURB] == "DQ") } {
  if { $array(N_NMA_DQ) == 0 } {
     WarningMessage "NOT SET: DQ steps"
     return 0
  }
  }
  if { ([GetValue $arrayname NMA_PERTURB] == "DQ") } {
  for { set n 1 } { $n <= $array(N_NMA_DQ) } { incr n } {
    if { $array(NMA_DQ,$n) == "" } {
     WarningMessage "NOT SET: normal mode dq"
     return 0
    }
  }
  }
  if {([GetValue $arrayname NMA_PERTURB] == "RMS")} {
  if { $array(NMA_RMSINC) == "" } {
     WarningMessage "NOT SET: RMS step"
     return 0
  }
  if { $array(NMA_MAXRMS) == "" } {
     WarningMessage "NOT SET: Max RMS distance"
     return 0
  }
  if { $array(TOG_NMA_CLASH) && $array(TOG_NMA_CLASH) == "" } {
     WarningMessage "NOT SET: Clash distance between atoms"
     return 0
  }
  if { $array(TOG_NMA_STRETCH) && $array(TOG_NMA_STRETCH) == "" } {
     WarningMessage "NOT SET: Stretch distance between atoms"
     return 0
  }
  }
  if { $array(TOG_NMA_FORCE) && $array(NMA_FORCE) == "" } {
     WarningMessage "NOT SET: Elastic Network Model force constant"
     return 0
  }
  if { $array(TOG_NMA_RADIUS) && $array(NMA_RADIUS) == "" } {
     WarningMessage "NOT SET: Elastic Network Model interaction radius"
     return 0
  }
  if { $array(TOG_NMA_COMBINATION) && $array(NMA_NCOMB) == "" } {
     WarningMessage "NOT SET: Number of modes to combine at a time"
     return 0
  }

  set array(INPUT_FILES) "NMA_XYZIN"
    if { ([GetValue $arrayname NMA_EIGEN] == "READ") } {
      append array(INPUT_FILES) " EIGENIN"
  }
  return 1
  }

#------------------------------------------------------------------------
 proc phaser_NMA_dq_list { arrayname counter_dqs } {
#------------------------------------------------------------------------
# The auto search extending frame code for adding dqs
  upvar #0 $arrayname array
  CreateLine line \
      label "    " \
      label "Perturb with DQ " widget NMA_DQ -oblig
  }

#------------------------------------------------------------------------
 proc phaser_NMA_mode_list { arrayname counter_modes } {
#------------------------------------------------------------------------
# The auto search extending frame code for adding modes
  upvar #0 $arrayname array
  CreateLine line \
      label "Perturb along mode " widget NMA_MODE -oblig 
  }

#------------------------------------------------------------------------
  proc phaser_NMA_task_window {arrayname} {
#------------------------------------------------------------------------

  upvar #0 $arrayname array
#  global configure

  if { [CreateTaskWindow $arrayname \
        "Normal Mode Analysis"\
        "Phaser" \
        [ list "Define data" \
               "Define perturbations" \
               "Output control" \
               "Expert parameters" ] ] == 0 } return

# SetProgramHelpFile phaser

#=PROTOCOL===============================================================

  OpenFolder protocol

  CreateTitleLine line TITLE


#=FILES==================================================================
# The "Define data" folder for Normal Mode Analysis

  OpenFolder 1 

  CreateInputFileLine line \
      "Select a PDB file" \
      "PDB file" \
      NMA_XYZIN DIR_NMA_XYZIN

  CreateLine line \
      widget TOG_NMA_EIGEN -toggleon\
      label "Eigenvalues and Eigenvectors" \
      message "Read or Write Eigenvalues and Eigenvectors" \
      widget NMA_EIGEN 

  CreateLine line \
      message "For output sol file" \
      label "Ensemble ID" -italic \
      widget ENSEMBLE_ID_NMA

  CreateLine line \
      message "For output sol file" \
      label "Similarity of output PDB files to target structure" -italic \
      widget ENSEMBLE_RMSID_OPTION_NMA \
      widget RMS_NMA -oblig
      
  CreateInputFileLine line \
      "Eigen File input MUST have been generated with an identical NMA Method" \
      "Eigen File" \
      EIGENIN DIR_EIGENIN \
      -toggle_display NMA_EIGEN open READ hide

#========================================================================
# The "Define perturbations" folder for Normal Mode Analysis

  OpenFolder 2 

  CreateLine line \
      label "Note: Mode 7 is the lowest non-trivial mode (after the 3 rotation and 3 translation modes)" -italic

  CreateExtendingFrame N_NMA_MODES phaser_NMA_mode_list \
      "Perturb along mode" \
      "Add a mode for perturbation" \
      [list NMA_MODE] 

  CreateLine line \
      widget TOG_NMA_COMBINATION -toggleon\
      label "Number of modes to combine at a time" \
      widget NMA_NCOMB

  CreateLine line \
      label "Perturb structure in " widget NMA_PERTURB \
      label " steps" 

  CreateLine line \
      label "    " \
      message "RMS deviation step size" \
      label "RMS step" widget NMA_RMSINC \
      label "Max RMS distance" widget NMA_MAXRMS \
      label "Direction" widget NMA_DIRECTION \
      toggle_display NMA_PERTURB open RMS 

  CreateLine line \
      label "    " \
      widget TOG_NMA_CLASH -toggleon \
      label "Perturbation stops when any two atoms come within" widget NMA_CLASH \
      label "Angstroms" \
      toggle_display NMA_PERTURB open RMS 

  CreateLine line \
      label "    " \
      widget TOG_NMA_STRETCH -toggleon \
      label "Perturbation stops when a bond stretches past" widget NMA_STRETCH \
      label "Angstroms" \
      toggle_display NMA_PERTURB open RMS 
  
  OpenSubFrame frame -toggle_display NMA_PERTURB open [list DQ ] hide

  CreateExtendingFrame N_NMA_DQ phaser_NMA_dq_list \
      "Perturb with dq" \
      "List DQ steps explicitly" \
      [list NMA_DQ] 

  CloseSubFrame


#========================================================================
# The "Output control" folder for Normal Mode Analysis

  OpenFolder 3 closed

  CreateLine line \
      message "VERBOSE: more information included in log file than by default" \
      widget TOG_VERBOSE \
      label "Verbose output to log file" \

  CreateLine line \
      message "DEBUG: debugging information included in log file than by default" \
      widget TOG_VERBOSE_EXTRA \
      label "Debug output to log file" 

  CreateLine line \
      message "Controls output of PDB files containing potential solutions.\
               Default depends on mode." \
      widget TOG_XYZOUT -toggleon \
      label "PDB file output" \
      widget XYZOUT_ONOFF


#========================================================================
# The "Expert parameters" folder for Normal Mode Analysis

  OpenFolder 4 closed

  CreateLine line \
      message "Controls NMA method" \
      widget TOG_NMA_METHOD -toggleon \
      label "Normal Mode Analysis Method" \
      widget NMA_METHOD  

  CreateLine line \
      label "    " \
      widget TOG_NMA_NRES -toggleon \
      label "Number of residues per block" \
      widget NMA_NRES  \
      toggle_display NMA_METHOD open RTB hide

  CreateLine line \
      label "    " \
      widget TOG_NMA_MAXB -toggleon \
      label "Maximum number of blocks" \
      widget NMA_MAXB \
      toggle_display NMA_METHOD open RTB hide
  CreateLine line \
      widget TOG_NMA_FORCE -toggleon \
      label "Elastic Network Model force constant" widget NMA_FORCE 

  CreateLine line \
      widget TOG_NMA_RADIUS -toggleon \
      label "Elastic Network Model interaction radius" widget NMA_RADIUS label "Angstroms" 

  }
